/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse AUTO_MOUSE

/////////////////////////////////////////////////////////////////////////////////

// Simple "Automatic Mouse". Simulates scanning the mouse over the full range of
// the screen with the X and Y scanning frequencies being different. TheEmu.

#define MOUSE_SPEED vec2(0.2,0.277777)
#define MOUSE_POS   vec2((1.0+cos(iGlobalTime*MOUSE_SPEED))*u_WindowSize/2.0)
#define MOUSE_PRESS vec2(0.0,0.0)
#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )

/////////////////////////////////////////////////////////////////////////////////

/*
Making a worley fractal is sorta weak, because each iteration generates entirely new
cells, unrelated to the "parent" cells. So you end up with what looks like a bunch of
layers overlayed, instead of a clean tessellation.

But that effect might just be what you're looking for. It has a soft, crumpled look.

Compare this to a similar fractal noise pattern based on just tessellating triangles:
https://www.shadertoy.com/view/MtSGWK

The triangle version looks much "sharper" and harder, because the edges ar retained
for each iteration. So it looks like a single surface, tessellated.

This worley version looks more "blurred", because edges are re-built each iteration.

*/

// -------------------------------------------------------
// parameters...


// 0 = no coloring; just see bump effect
// 1 = stylized by distance
// 2 = reveal raw distance (typical worley)
#define COLOR_MODE 1


const int iterationCount = 10;

// normally this should be 0.5 to cut cells in half each iteration.
// small amounts are cool too though like 0.1.
// greater than 1 doesn't make much sense considering the algo, and <0 is just nonsense.
const float iterationSizeFactor = .5;

const float iterationNormalInfluenceFactor = 0.79;

const float iterationColorInfluenceFactor = 0.68;





// -------------------------------------------------------
// utilities
// c64 palette because why not.
vec3 color0 = vec3(0,0,0);// black
vec3 color1 = vec3(1,1,1);// white
vec3 color2 = vec3(0.41,0.22,0.17);// red
vec3 color3 = vec3(0.44,0.64,0.70);// cyan
vec3 color4 = vec3(0.44,0.24,0.53);// violet
vec3 color5 = vec3(0.35,0.55,0.26);// green
vec3 color6 = vec3(0.21,0.16,0.47);// blue
vec3 color7 = vec3(0.72,0.78,0.44);// yellow
vec3 color8 = vec3(0.44,0.31,0.15);// orange
vec3 color9 = vec3(0.26,0.22,0);// brown
vec3 colorA = vec3(0.60,0.40,0.35);// light red
vec3 colorB = vec3(0.27,0.27,0.27);// grey1
vec3 colorC = vec3(0.42,0.42,0.42);// grey2
vec3 colorD = vec3(0.60,0.82,0.52);// light green
vec3 colorE = vec3(0.42,0.37,0.71);// light blue
vec3 colorF = vec3(0.58,0.58,0.58);// grey3
vec3 getPalette(int i)
{
    if(i == 0) return color6;
    if(i == 1) return color3;
    if(i == 2) return color5;
    if(i == 3) return color9;
    if(i == 4) return color7;
    return color8;
}



float nsin(float a)
{
    return (sin(a)+1.)/2.;
}
float rand(float n)
{
 	return fract(cos(n*89.42)*343.42);
}
vec2 rand(vec2 n)
{
 	return vec2(rand(n.x*23.62-300.0+n.y*34.35),rand(n.x*45.13+256.0+n.y*38.89)); 
}

// returns (dx, dy, distance)
vec3 worley(vec2 n,float s)
{
    vec3 ret = vec3(s * 10.);
    // look in 9 cells (n, plus 8 surrounding)
    for(int x = -1;x<2;x++)
    {
        for(int y = -1;y<2;y++)
        {
            vec2 xy = vec2(x,y);// xy can be thought of as both # of cells distance to n, and 
            vec2 cellIndex = floor(n/s) + xy;
            vec2 worleyPoint = rand(cellIndex);// random point in this cell (0-1)
            worleyPoint += xy - fract(n/s);// turn it into distance to n. ;
            float d = length(worleyPoint) * s;
            if(d < ret.z)
                ret = vec3(worleyPoint, d);
        }
    }
    return ret;
}


vec4 applyLighting(vec4 inpColor, vec2 uv, vec3 normal, vec3 LightPos, vec4 LightColor, vec4 AmbientColor)
{
    if(distance(uv.xy, LightPos.xy) < 0.01) return vec4(1.,0.,0.,1.);
    vec3 LightDir = vec3(LightPos.xy - uv, LightPos.z);
    float D = length(LightDir);// distance for attenuation
    vec3 N = normalize(normal);
    vec3 L = normalize(LightDir);
    vec3 Diffuse = (LightColor.rgb * LightColor.a) * max(dot(N, L), 0.0);
    vec3 Ambient = AmbientColor.rgb * AmbientColor.a;
    vec3 Intensity = Ambient + Diffuse;
    vec3 FinalColor = inpColor.rgb * Intensity;
    return vec4(FinalColor, inpColor.a);
}

void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
    vec2 scrollOffset = vec2(iGlobalTime * iResolution.x / 17., 0.0);
    vec2 uv = fragCoord.xy + scrollOffset;
    
    vec2 lightPosXY = vec2(nsin(iGlobalTime * 1.66), nsin(iGlobalTime * 1.99));
    lightPosXY = (lightPosXY * 0.75) + 0.125;// not so big movement
    lightPosXY *= iResolution.xy;
    lightPosXY += scrollOffset;
    if(iMouse.z > 0.)
	    lightPosXY = iMouse.xy + scrollOffset;
    
    fragColor = vec4(vec3(0.5), 1.0);// background

    // compute the normal for this fragment by subdividing.
    float wsize = iResolution.x;
    vec2 normal = vec2(0.);
    float influenceFactor = 1.0;
    
    float colorMix = 1.0;
    
    for(int i = 0; i < iterationCount; ++ i)
    {
        vec3 w = worley(uv, wsize);
		normal.xy += influenceFactor * w.xy;
        wsize *= iterationSizeFactor;
        influenceFactor *= iterationNormalInfluenceFactor;

        float tintAmt = w.z / wsize;
        
        // i don't really understand why this looks so much better at low size factors.
        if(iterationSizeFactor < 0.5)
	        tintAmt = tintAmt * iterationSizeFactor;
        
#if COLOR_MODE == 1
        fragColor = vec4(mix(fragColor.rgb, getPalette(i), tintAmt * colorMix), 1.0);
#endif
#if COLOR_MODE == 2
        fragColor = vec4(vec3(tintAmt), 1.0);
#endif
        colorMix *= iterationColorInfluenceFactor;
    }
    
    // lighting
#if COLOR_MODE != 2
    vec3 lightPos = vec3(lightPosXY, iResolution.x / 2.);
    vec4 lightColor = vec4(0.95,0.9,0.9,1.);
    vec4 ambientColor = vec4(0.5,0.5,0.9,1.);
	fragColor = applyLighting(fragColor, uv, vec3(normal, 2.5), lightPos, lightColor, ambientColor);
    
    // show light position.
    if(distance(uv, lightPosXY) < iResolution.x / 75.)
        fragColor = vec4(1.0,1.0,0.,1.0);
#endif
}

// -------------------------------------------------------

void main( void )
{ mainImage ( gl_FragColor, gl_FragCoord.xy );
}
